<?php
class Database {
    private $connection;
    
    public function __construct() {
        try {
            $this->connection = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
        } catch(PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            die("خطا در اتصال به دیتابیس");
        }
    }
    
    public function getConnection() {
        return $this->connection;
    }
    
    // ایجاد جداول مورد نیاز
    public function createTables() {
        $queries = [
            // جدول کاربران
            "CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                telegram_id BIGINT UNIQUE NOT NULL,
                username VARCHAR(255),
                first_name VARCHAR(255),
                last_name VARCHAR(255),
                balance INT DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                is_active BOOLEAN DEFAULT TRUE
            )",
            
            // جدول تراکنش‌ها
            "CREATE TABLE IF NOT EXISTS transactions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                type ENUM('deposit', 'withdraw', 'order') NOT NULL,
                amount INT NOT NULL,
                description TEXT,
                status ENUM('pending', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
                reference_id VARCHAR(255),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            )",
            
            // جدول سفارشات
            "CREATE TABLE IF NOT EXISTS orders (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                bot_type VARCHAR(50) NOT NULL,
                bot_token TEXT NOT NULL,
                amount INT NOT NULL,
                status ENUM('pending', 'processing', 'completed', 'cancelled') DEFAULT 'pending',
                admin_notes TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            )",
            
            // جدول درخواست‌های مشاوره
            "CREATE TABLE IF NOT EXISTS consultations (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                subject VARCHAR(255) NOT NULL,
                description TEXT NOT NULL,
                contact_info VARCHAR(255) NOT NULL,
                status ENUM('pending', 'replied', 'closed') DEFAULT 'pending',
                admin_reply TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            )",
            
            // جدول لاگ‌ها
            "CREATE TABLE IF NOT EXISTS logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT,
                action VARCHAR(100) NOT NULL,
                details TEXT,
                ip_address VARCHAR(45),
                user_agent TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
            )"
        ];
        
        foreach ($queries as $query) {
            try {
                $this->connection->exec($query);
            } catch(PDOException $e) {
                error_log("Table creation failed: " . $e->getMessage());
            }
        }
    }
    
    // دریافت یا ایجاد کاربر
    public function getUser($telegram_id, $user_data = null) {
        $stmt = $this->connection->prepare("SELECT * FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegram_id]);
        $user = $stmt->fetch();
        
        if (!$user && $user_data) {
            $stmt = $this->connection->prepare("
                INSERT INTO users (telegram_id, username, first_name, last_name) 
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([
                $telegram_id,
                $user_data['username'] ?? null,
                $user_data['first_name'] ?? null,
                $user_data['last_name'] ?? null
            ]);
            return $this->getUser($telegram_id);
        }
        
        return $user;
    }
    
    // به‌روزرسانی موجودی کاربر
    public function updateBalance($user_id, $amount, $type = 'deposit') {
        $this->connection->beginTransaction();
        
        try {
            // به‌روزرسانی موجودی
            $stmt = $this->connection->prepare("
                UPDATE users SET balance = balance + ? WHERE id = ?
            ");
            $stmt->execute([$amount, $user_id]);
            
            // ثبت تراکنش
            $stmt = $this->connection->prepare("
                INSERT INTO transactions (user_id, type, amount, status) 
                VALUES (?, ?, ?, 'completed')
            ");
            $stmt->execute([$user_id, $type, $amount]);
            
            $this->connection->commit();
            return true;
        } catch(Exception $e) {
            $this->connection->rollback();
            error_log("Balance update failed: " . $e->getMessage());
            return false;
        }
    }
    
    // ثبت سفارش
    public function createOrder($user_id, $bot_type, $bot_token, $amount) {
        $stmt = $this->connection->prepare("
            INSERT INTO orders (user_id, bot_type, bot_token, amount) 
            VALUES (?, ?, ?, ?)
        ");
        return $stmt->execute([$user_id, $bot_type, $bot_token, $amount]);
    }
    
    // ثبت درخواست مشاوره
    public function createConsultation($user_id, $subject, $description, $contact_info) {
        $stmt = $this->connection->prepare("
            INSERT INTO consultations (user_id, subject, description, contact_info) 
            VALUES (?, ?, ?, ?)
        ");
        return $stmt->execute([$user_id, $subject, $description, $contact_info]);
    }
    
    // دریافت تراکنش‌های کاربر
    public function getUserTransactions($user_id, $limit = 10) {
        $stmt = $this->connection->prepare("
            SELECT * FROM transactions 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$user_id, $limit]);
        return $stmt->fetchAll();
    }
    
    // دریافت سفارشات جدید برای ادمین
    public function getNewOrders() {
        $stmt = $this->connection->prepare("
            SELECT o.*, u.telegram_id, u.username, u.first_name 
            FROM orders o 
            JOIN users u ON o.user_id = u.id 
            WHERE o.status = 'pending' 
            ORDER BY o.created_at DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    // دریافت درخواست‌های مشاوره جدید
    public function getNewConsultations() {
        $stmt = $this->connection->prepare("
            SELECT c.*, u.telegram_id, u.username, u.first_name 
            FROM consultations c 
            JOIN users u ON c.user_id = u.id 
            WHERE c.status = 'pending' 
            ORDER BY c.created_at DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    // ثبت لاگ
    public function log($user_id, $action, $details = null) {
        $stmt = $this->connection->prepare("
            INSERT INTO logs (user_id, action, details, ip_address, user_agent) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $user_id,
            $action,
            $details,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    }
}
?>
